import numpy as np
from scipy.signal import welch, stft, coherence, hann
import matplotlib.pyplot as plt
import os

def compute_welch_psd(values, fs=200.0, nperseg=None):
    if nperseg is None:
        nperseg = int(fs * 10)  # 10-second window default
    freqs, psd = welch(values, fs=fs, window='hann', nperseg=nperseg, noverlap=int(nperseg/2))
    return freqs, psd

def compute_stft(values, fs=200.0, nperseg=None):
    if nperseg is None:
        nperseg = int(fs * 4)
    f, t, Zxx = stft(values, fs=fs, window='hann', nperseg=nperseg, noverlap=int(nperseg/2))
    return f, t, Zxx

def compute_coherence(x, y, fs=200.0, nperseg=None):
    if nperseg is None:
        nperseg = int(fs * 10)
    f, Cxy = coherence(x, y, fs=fs, window='hann', nperseg=nperseg, noverlap=int(nperseg/2))
    return f, Cxy

def plot_psd(freqs, psd, title='PSD Estimate', outpath=None):
    plt.figure(figsize=(8,4))
    plt.semilogy(freqs, psd)
    plt.xlim(0, 60)
    plt.xlabel('Frequency (Hz)')
    plt.ylabel('PSD (V^2/Hz)')
    plt.title(title)
    plt.grid(True)
    if outpath:
        plt.savefig(outpath, bbox_inches='tight')
        plt.close()
    else:
        plt.show()

def save_spectrogram(f, t, Zxx, outpath):
    plt.figure(figsize=(10,4))
    plt.pcolormesh(t, f, np.abs(Zxx), shading='gouraud')
    plt.ylabel('Frequency [Hz]')
    plt.xlabel('Time [sec]')
    plt.ylim(0, 60)
    plt.colorbar(label='Magnitude')
    if outpath:
        plt.savefig(outpath, bbox_inches='tight')
        plt.close()
    else:
        plt.show()

if __name__ == '__main__':
    import argparse, numpy as _np
    p = argparse.ArgumentParser()
    p.add_argument('npz', help='Path to ingested .npz file (times,values)')
    p.add_argument('--fs', type=float, default=200.0)
    args = p.parse_args()
    data = _np.load(args.npz)
    values = data['values']
    freqs, psd = compute_welch_psd(values, fs=args.fs)
    plot_psd(freqs, psd, outpath=args.npz + '.psd.png')
    f,t,Z = compute_stft(values, fs=args.fs)
    save_spectrogram(f,t,Z, outpath=args.npz + '.spec.png')
